let isRunning = false;
let currentBatchTraceId = null;
let currentIndex = 0;

function sleep(ms) {
  return new Promise(resolve => setTimeout(resolve, ms));
}

function randInt(min, max) {
  const a = Math.min(min, max);
  const b = Math.max(min, max);
  return a + Math.floor(Math.random() * (b - a + 1));
}

function truncateText(s, maxLen) {
  if (!s) return '';
  const t = String(s).replace(/\s+/g, ' ').trim();
  return t.length > maxLen ? t.slice(0, maxLen) + '…' : t;
}

async function log(level, message, extra) {
  try {
    await chrome.runtime.sendMessage({ type: 'TK_SYNC_LOG', level, message, extra });
  } catch {
    // ignore
  }
}

function textIncludesAny(hay, needles) {
  const s = (hay || '').toLowerCase();
  return needles.some(n => s.includes(String(n).toLowerCase()));
}

function normalizeIdForInviteInput(s) {
  // 目标：兼容 @unique_id / unique_id / URL 等输入。抽屉里一般接受唯一ID（不带 @）。
  // 这里尽量“温和处理”，避免破坏原值。
  const raw = String(s || '').trim();
  if (!raw) return '';
  // 去掉 TikTok profile URL 的前缀（保留 @xxx 或 xxx）
  const m = raw.match(/\/@([^/?#\s]+)/i);
  const v = m ? m[1] : raw;
  return v.replace(/^@/, '').trim();
}

function waitForElement(selectors, timeoutMs) {
  return new Promise((resolve, reject) => {
    const start = Date.now();

    function find() {
      for (const sel of selectors) {
        try {
          const el = document.querySelector(sel);
          if (el) return el;
        } catch {
          // ignore invalid selector
        }
      }
      return null;
    }

    const immediate = find();
    if (immediate) return resolve(immediate);

    const obs = new MutationObserver(() => {
      const el = find();
      if (el) {
        cleanup();
        resolve(el);
      }
    });

    obs.observe(document.documentElement, { childList: true, subtree: true });

    const timer = setInterval(() => {
      const el = find();
      if (el) {
        cleanup();
        resolve(el);
        return;
      }
      if (Date.now() - start > timeoutMs) {
        cleanup();
        reject(new Error('waitForElement timeout'));
      }
    }, 300);

    function cleanup() {
      obs.disconnect();
      clearInterval(timer);
    }
  });
}

function setNativeValue(input, value) {
  // 兼容 input/textarea 与 contenteditable（TikTok 后台会把“搜索框”实现成可编辑 div）
  const isCe = !!(input && (input.isContentEditable || String(input.getAttribute?.('contenteditable') || '').toLowerCase() === 'true'));
  if (isCe) {
    const lastText = input.textContent || '';
    input.textContent = String(value ?? '');
    // 尽量触发框架监听
    input.dispatchEvent(new Event('input', { bubbles: true }));
    if (lastText !== input.textContent) input.dispatchEvent(new Event('change', { bubbles: true }));
    return;
  }

  const lastValue = input.value;
  input.value = value;
  const event = new Event('input', { bubbles: true });
  // React/Vue 受控输入常见的 setter hack
  const tracker = input._valueTracker;
  if (tracker) tracker.setValue(lastValue);
  input.dispatchEvent(event);
  input.dispatchEvent(new Event('change', { bubbles: true }));
}

function isVisible(el) {
  if (!el) return false;
  const rect = el.getBoundingClientRect?.();
  if (!rect) return true;
  if (rect.width <= 0 || rect.height <= 0) return false;
  const style = window.getComputedStyle?.(el);
  if (!style) return true;
  if (style.display === 'none' || style.visibility === 'hidden' || style.opacity === '0') return false;
  return true;
}

const INVITE_HOST_BTN_SELECTOR = 'button[data-id="add-host-btn"]';
const _inviteHostBtnMeta = new WeakMap();

function isOverviewPageUrl(url) {
  try {
    const u = new URL(url || location.href);
    return u.hostname === 'live-backstage.tiktok.com' && u.pathname.startsWith('/portal/overview');
  } catch {
    return false;
  }
}

function _markInviteHostBtn(el, meta) {
  try {
    if (el) _inviteHostBtnMeta.set(el, meta || {});
  } catch {
    // ignore
  }
  return el;
}

function _getInviteHostBtnMeta(el) {
  try {
    return _inviteHostBtnMeta.get(el) || null;
  } catch {
    return null;
  }
}

function _findInviteHostButtonInDoc(doc) {
  if (!doc) return null;

  // 1) 优先稳定属性
  try {
    const direct = doc.querySelector(INVITE_HOST_BTN_SELECTOR);
    if (direct) return _markInviteHostBtn(direct, { strategy: 'selector:data-id', selector: INVITE_HOST_BTN_SELECTOR, inFrame: doc !== document });
  } catch {
    // ignore
  }

  // 2) 文案兜底（仅按你要求的“邀请主播”）
  try {
    const nodes = Array.from(doc.querySelectorAll('button, [role="button"]'));
    for (const n of nodes) {
      if (!isVisible(n)) continue;
      const t = (n.innerText || n.textContent || '').replace(/\s+/g, ' ').trim();
      if (!t) continue;
      if (t.includes('邀请主播')) return _markInviteHostBtn(n, { strategy: 'text:invite_host', selector: 'button,[role="button"]', inFrame: doc !== document });
    }
  } catch {
    // ignore
  }

  return null;
}

// 新增：按需求的接口（返回 Element 或 null）
function findInviteHostButton() {
  // 主文档
  const main = _findInviteHostButtonInDoc(document);
  if (main) return main;

  // 同源 iframe 兜底（跨域 iframe 无法访问：catch 后跳过）
  const iframes = Array.from(document.querySelectorAll('iframe'));
  for (let i = 0; i < iframes.length; i += 1) {
    const fr = iframes[i];
    try {
      const doc = fr.contentDocument;
      if (!doc) continue;
      const hit = _findInviteHostButtonInDoc(doc);
      if (hit) {
        return _markInviteHostBtn(hit, { ...(_getInviteHostBtnMeta(hit) || {}), inFrame: true, frameIndex: i });
      }
    } catch {
      // 跨域：无法访问，跳过
      continue;
    }
  }

  return null;
}

function _dispatchMouseSequence(el) {
  if (!el) return;
  const rect = el.getBoundingClientRect?.();
  const clientX = rect ? Math.floor(rect.left + rect.width / 2) : 0;
  const clientY = rect ? Math.floor(rect.top + rect.height / 2) : 0;
  const common = { bubbles: true, cancelable: true, view: window, button: 0, buttons: 1, clientX, clientY };
  try { el.dispatchEvent(new MouseEvent('mousemove', common)); } catch {}
  try { el.dispatchEvent(new MouseEvent('mouseover', common)); } catch {}
  try { el.dispatchEvent(new MouseEvent('mouseenter', common)); } catch {}
  try { el.dispatchEvent(new MouseEvent('mousedown', common)); } catch {}
  try { el.dispatchEvent(new MouseEvent('mouseup', common)); } catch {}
  try { el.dispatchEvent(new MouseEvent('click', common)); } catch {}
}

// 新增：按需求的接口（更“真人点击”）
function clickInviteHostButton(el) {
  if (!el) throw new Error('invite_host_button_missing');
  try {
    el.scrollIntoView({ block: 'center', inline: 'center' });
  } catch {
    // ignore
  }

  try { el.focus?.(); } catch {}

  // 先走原生 click；不稳定再补 mouse 事件序列
  try { el.click(); } catch {}
  _dispatchMouseSequence(el);
}

function findClickableByText(root, texts) {
  const wanted = Array.isArray(texts) ? texts : [texts];
  const nodes = Array.from((root || document).querySelectorAll('button, [role="button"], [role="menuitem"], a[role="button"], a[href]'));
  for (const n of nodes) {
    if (!isVisible(n)) continue;
    const raw = (n.innerText || n.textContent || '').replace(/\s+/g, ' ').trim();
    if (!raw) continue;
    const t = raw.toLowerCase();
    if (wanted.some(w => w && t.includes(String(w).toLowerCase()))) return n;
  }
  return null;
}

function getSearchInputSelectors() {
  // TikTok 后台经常改 placeholder / role，这里用更“宽松”的多候选选择器：
  // - type=search / role=searchbox / aria-label / name / data-testid
  // - 同时保留旧的 placeholder 逻辑
  return [
    'input[type="search"]',
    'input[role="searchbox"]',
    '[role="searchbox"] input',
    'input[aria-label*="Search" i]',
    'input[placeholder*="Search" i]',
    'input[placeholder*="搜索"]',
    'input[name*="search" i]',
    'input[id*="search" i]',
    'input[class*="search" i]',
    'input[data-testid*="search" i]',
    '[data-testid*="search" i] input',
    // 某些实现用 combobox 承载搜索框
    '[role="combobox"] input',
    'input[role="combobox"]',
    '[role="combobox"][contenteditable="true"]',
    '[role="combobox"] [contenteditable="true"]',
    // 有些实现直接用 textarea（可粘贴多行或支持 IME）
    'textarea[aria-label*="Search" i]',
    'textarea[placeholder*="Search" i]',
    'textarea[placeholder*="搜索"]',
    // contenteditable / textbox
    '[role="searchbox"][contenteditable="true"]',
    '[role="textbox"][contenteditable="true"]',
    '[contenteditable="true"][aria-label*="Search" i]',
    '[contenteditable="true"][placeholder*="Search" i]',
    '[contenteditable="true"][placeholder*="搜索"]'
  ];
}

function isEditableControl(el) {
  if (!el) return false;
  const tag = (el.tagName || '').toLowerCase();
  if (tag === 'input' || tag === 'textarea') return true;
  if (el.isContentEditable) return true;
  const ce = String(el.getAttribute?.('contenteditable') || '').toLowerCase();
  return ce === 'true';
}

function findSearchInputNow(root) {
  const r = root || document;
  const selectors = getSearchInputSelectors();
  for (const sel of selectors) {
    try {
      const el = r.querySelector(sel);
      if (el && isVisible(el)) return el;
    } catch {
      // ignore invalid selector
    }
  }

  // 兜底：在可见的可编辑控件中用属性启发式识别“搜索框”
  const candidates = Array.from(r.querySelectorAll('input, textarea, [contenteditable="true"]'));
  const cand = candidates.find(el => {
    if (!isVisible(el)) return false;
    const tag = (el.tagName || '').toLowerCase();
    if (tag === 'input') {
      const type = (el.getAttribute('type') || '').toLowerCase();
      if (type && !['text', 'search'].includes(type)) return false;
    }
    const ph = el.getAttribute('placeholder') || '';
    const aria = el.getAttribute('aria-label') || '';
    const role = el.getAttribute('role') || '';
    const name = el.getAttribute('name') || '';
    const id = el.getAttribute('id') || '';
    const dt = el.getAttribute('data-testid') || '';
    const cls = el.getAttribute('class') || '';
    const hint = [ph, aria, role, name, id, dt, cls].join(' ');
    return textIncludesAny(hint, ['search', '搜索', '查询', '查找']);
  });
  return cand || null;
}

function detectSyncUiAvailable() {
  // 邀请入口判定依赖 detectInviteFlowAvailable（函数声明可提升）
  const invite = detectInviteFlowAvailable();
  const search = !!findSearchInputNow(document);
  return { invite, search };
}

async function ensureSyncUiContext(prefer) {
  // 目标：尽量从“任意页面”切到可执行状态（存在邀请入口或搜索框）
  // prefer: 'invite' | 'search' | undefined
  const initial = detectSyncUiAvailable();
  if (prefer === 'invite' && initial.invite) return { mode: 'invite', reason: 'invite_already_available' };
  if (prefer === 'search' && initial.search) return { mode: 'search', reason: 'search_already_available' };
  if (initial.invite) return { mode: 'invite', reason: 'invite_available' };
  if (initial.search) return { mode: 'search', reason: 'search_available' };

  const keywordTexts = [
    // CN
    '邀请主播', '招募主播', '邀请达人', '招募达人', '邀请创作者', '招募创作者', '主播招募', '达人招募', '招募', '邀请',
    // EN
    'Invite', 'Recruit', 'Creator', 'Creators', 'Host', 'Hosts'
  ];
  const hrefKeywords = ['recruit', 'invite', 'creator', 'creators', 'host', 'hosts', 'talent', 'affiliate'];

  // 先尝试点击“看起来像入口/菜单”的链接（同域）
  const links = Array.from(document.querySelectorAll('a[href]')).filter(a => {
    if (!isVisible(a)) return false;
    const href = a.getAttribute('href') || '';
    if (!href) return false;
    if (/^https?:\/\//i.test(href) && !href.startsWith(location.origin)) return false;
    const text = (a.innerText || a.textContent || '').replace(/\s+/g, ' ').trim();
    const hay = `${href} ${text}`.toLowerCase();
    return hrefKeywords.some(k => hay.includes(k));
  });

  for (const a of links.slice(0, 8)) {
    try {
      a.click();
      await sleep(800);
      const after = detectSyncUiAvailable();
      if (after.invite) return { mode: 'invite', reason: 'nav_href_click_invite' };
      if (after.search) return { mode: 'search', reason: 'nav_href_click_search' };
    } catch {
      // ignore
    }
  }

  // 再尝试按文本点击按钮/菜单（避免误点过多，限量尝试）
  const clickables = Array.from(document.querySelectorAll('button, [role="button"], a[role="button"], a[href]'));
  const byText = clickables.filter(n => {
    if (!isVisible(n)) return false;
    const raw = (n.innerText || n.textContent || '').replace(/\s+/g, ' ').trim();
    if (!raw) return false;
    const t = raw.toLowerCase();
    return keywordTexts.some(k => t.includes(String(k).toLowerCase()));
  });

  for (const n of byText.slice(0, 8)) {
    try {
      n.click();
      await sleep(800);
      const after = detectSyncUiAvailable();
      if (after.invite) return { mode: 'invite', reason: 'nav_text_click_invite' };
      if (after.search) return { mode: 'search', reason: 'nav_text_click_search' };
    } catch {
      // ignore
    }
  }

  const final = detectSyncUiAvailable();
  if (final.invite) return { mode: 'invite', reason: 'invite_available_after_nav' };
  if (final.search) return { mode: 'search', reason: 'search_available_after_nav' };
  return { mode: 'none', reason: 'no_invite_or_search_ui' };
}

function findDialogLikeRoot() {
  // 常见：role=dialog / drawer 容器。找“可见且文本量较多”的候选。
  const candidates = Array.from(document.querySelectorAll('[role="dialog"], [aria-modal="true"], .semi-modal, .semi-modal-content'));
  const visible = candidates.filter(isVisible);
  if (visible.length) return visible[visible.length - 1]; // 取最上层
  // fallback：页面里出现“添加主播/下一步”的区域
  const all = Array.from(document.querySelectorAll('div,section,main,aside'));
  const hit = all.find(el => isVisible(el) && textIncludesAny(el.innerText || '', ['添加主播', '下一步', '邀请主播']));
  return hit || null;
}

function findMultilineInput(root) {
  // 你描述的是“多行输入框”，优先找 textarea，其次找 aria-multiline 或 contenteditable。
  const r = root || document;
  const ta = Array.from(r.querySelectorAll('textarea')).find(isVisible);
  if (ta) return ta;
  const ariaMulti = Array.from(r.querySelectorAll('[aria-multiline="true"], [contenteditable="true"]')).find(isVisible);
  if (ariaMulti) return ariaMulti;
  // 有些实现用 input 但支持换行粘贴；兜底取一个明显的输入框
  const inp = Array.from(r.querySelectorAll('input[type="text"], input:not([type])')).find(isVisible);
  return inp || null;
}

function extractSnippetAround(text, needle, radius) {
  const s = String(text || '');
  const n = String(needle || '');
  if (!s || !n) return null;
  const idx = s.toLowerCase().indexOf(n.toLowerCase());
  if (idx < 0) return null;
  const r = Number.isFinite(radius) ? radius : 120;
  const start = Math.max(0, idx - r);
  const end = Math.min(s.length, idx + n.length + r);
  return s.slice(start, end).replace(/\s+/g, ' ').trim();
}

async function openInviteDrawer() {
  const onOverview = isOverviewPageUrl(location.href);
  await log('debug', 'openInviteDrawer: start', { url: location.href, onOverview });

  // overview：严格按稳定 selector + 异步等待 + 抽屉判定 + 失败重试
  if (onOverview) {
    const timeoutMs = 15_000;
    const start = Date.now();

    const waitBtn = () =>
      new Promise((resolve, reject) => {
        const tick = () => {
          const el = findInviteHostButton();
          if (el) return el;
          return null;
        };

        const immediate = tick();
        if (immediate) return resolve(immediate);

        const obs = new MutationObserver(() => {
          const el = tick();
          if (el) {
            cleanup();
            resolve(el);
          }
        });

        obs.observe(document.documentElement, { childList: true, subtree: true });

        const timer = setInterval(() => {
          const el = tick();
          if (el) {
            cleanup();
            resolve(el);
            return;
          }
          if (Date.now() - start > timeoutMs) {
            cleanup();
            reject(new Error('invite_host_button_wait_timeout'));
          }
        }, 250);

        function cleanup() {
          obs.disconnect();
          clearInterval(timer);
        }
      });

    const waitDrawer = (drawerTimeoutMs) =>
      new Promise((resolve, reject) => {
        const t0 = Date.now();
        const isOpen = () => {
          const dlg = findDialogLikeRoot();
          if (dlg && isVisible(dlg)) return dlg;
          // 额外兜底：drawer/modal 常见 class/属性（不依赖具体实现）
          const candidates = Array.from(document.querySelectorAll(
            '[role="dialog"],[aria-modal="true"],[data-testid*="drawer" i],[class*="drawer" i],[class*="modal" i],.semi-modal,.semi-modal-content'
          ));
          const hit = candidates.filter(isVisible);
          return hit.length ? hit[hit.length - 1] : null;
        };

        const immediate = isOpen();
        if (immediate) return resolve(immediate);

        const obs = new MutationObserver(() => {
          const dlg = isOpen();
          if (dlg) {
            cleanup();
            resolve(dlg);
          }
        });
        obs.observe(document.documentElement, { childList: true, subtree: true, attributes: true });

        const timer = setInterval(() => {
          const dlg = isOpen();
          if (dlg) {
            cleanup();
            resolve(dlg);
            return;
          }
          if (Date.now() - t0 > drawerTimeoutMs) {
            cleanup();
            reject(new Error('invite_drawer_wait_timeout'));
          }
        }, 200);

        function cleanup() {
          obs.disconnect();
          clearInterval(timer);
        }
      });

    let btn;
    try {
      btn = await waitBtn();
    } catch (e) {
      await log('debug', 'openInviteDrawer: button not found (timeout)', { err: String(e), timeoutMs });
      throw e;
    }

    const meta = _getInviteHostBtnMeta(btn);
    await log('debug', 'openInviteDrawer: button found', { meta, text: truncateText(btn.innerText || btn.textContent || '', 80) });

    const doClickOnce = async (attempt) => {
      try {
        clickInviteHostButton(btn);
        await log('debug', 'openInviteDrawer: clicked', { attempt, meta });
      } catch (e) {
        await log('debug', 'openInviteDrawer: click failed', { attempt, err: String(e), meta });
      }
      // 点击后等待抽屉出现
      const dlg = await waitDrawer(6_000);
      return dlg;
    };

    try {
      await doClickOnce(1);
      await log('debug', 'openInviteDrawer: drawer detected', { attempt: 1 });
    } catch (e1) {
      await log('debug', 'openInviteDrawer: drawer not detected, retry click once', { err: String(e1) });
      await sleep(400);
      await doClickOnce(2);
      await log('debug', 'openInviteDrawer: drawer detected', { attempt: 2 });
    }

    await sleep(300);
    return;
  }

  // 非 overview：保持原有策略（文案覆盖更广），但用更“真人”的点击方式
  const btn = findClickableByText(document, [
    '邀请主播', '邀请达人', '邀请创作者',
    '招募主播', '招募达人', '招募创作者',
    'Invite', 'Recruit'
  ]);
  if (!btn) {
    await log('debug', 'openInviteDrawer: invite button not found (non-overview)', { url: location.href });
    throw new Error('invite_button_not_found');
  }

  try {
    clickInviteHostButton(btn);
  } catch {
    btn.click();
  }

  await waitForElement(['[role="dialog"]', '[aria-modal="true"]', 'textarea'], 30_000);
  await sleep(300);
}

async function fillDrawerAndNext(ids) {
  const dlg = findDialogLikeRoot();
  if (!dlg) throw new Error('drawer_not_found');

  // 确保抽屉内确实是“添加主播”步骤（但不强依赖）
  const dlgText = dlg.innerText || '';
  if (!textIncludesAny(dlgText, ['添加主播', '主播', '达人', 'invite'])) {
    // 仍继续尝试（有些页面标题不含该文案）
  }

  const input = findMultilineInput(dlg);
  if (!input) throw new Error('drawer_multiline_input_not_found');

  const normalized = (ids || []).map(normalizeIdForInviteInput).filter(Boolean);
  if (!normalized.length) throw new Error('empty_ids');

  const payload = normalized.join('\n');

  try {
    input.focus?.();
  } catch {
    // ignore
  }
  if (input.tagName && input.tagName.toLowerCase() === 'textarea') {
    setNativeValue(input, payload);
  } else if (input.isContentEditable) {
    input.textContent = payload;
    input.dispatchEvent(new Event('input', { bubbles: true }));
    input.dispatchEvent(new Event('change', { bubbles: true }));
  } else {
    // input[type=text] 兜底
    setNativeValue(input, payload);
  }

  // 点击“下一步”
  const nextBtn = findClickableByText(dlg, ['下一步', 'Next']);
  if (!nextBtn) throw new Error('next_button_not_found');
  nextBtn.click();

  // 等待结果区域渲染：优先等“下一步”按钮消失或出现明显结果关键词
  const t0 = Date.now();
  while (Date.now() - t0 < 30_000) {
    const cur = findDialogLikeRoot();
    if (!cur) break;
    const txt = cur.innerText || '';
    const hasResultHint = textIncludesAny(txt, ['结果', '符合', '不符合', '可邀', '不可', '失败', '未找到', 'No results', 'not found']);
    const stillHasNext = !!findClickableByText(cur, ['下一步', 'Next']);
    if (hasResultHint || !stillHasNext) break;
    await sleep(300);
  }

  return findDialogLikeRoot() || dlg;
}

function parseInviteResultsFromDrawer(dlg, ids) {
  const text = (dlg && (dlg.innerText || dlg.textContent)) ? String(dlg.innerText || dlg.textContent) : '';
  const out = [];
  const normalized = (ids || []).map(normalizeIdForInviteInput).filter(Boolean);

  for (const id of normalized) {
    const snip = extractSnippetAround(text, id, 160);
    // 粗粒度归一：尽可能把“可邀/不可邀/未找到”映射到服务端可接受的 norm
    let norm = 'unknown_error';
    if (textIncludesAny(snip || '', ['未找到', 'not found', 'no results', '无结果'])) norm = 'not_found';
    else if (textIncludesAny(snip || '', ['不可', '不符合', '受限', 'restricted', '违规', '冻结', 'suspend', 'banned'])) norm = 'restricted';
    else if (textIncludesAny(snip || '', ['可邀', '可以', '符合', 'eligible', 'success', '已找到'])) norm = 'active';

    out.push({
      id,
      tiktok_status_norm: norm,
      snippet: snip || truncateText(text, 240)
    });
  }
  return out;
}

async function closeDrawerIfPossible() {
  const dlg = findDialogLikeRoot();
  if (!dlg) return;

  // 尝试点“关闭/取消/×”
  const closeBtn =
    dlg.querySelector('[aria-label*="close" i], [aria-label*="关闭"], [data-testid*="close" i]') ||
    findClickableByText(dlg, ['关闭', '取消', 'Cancel']);
  if (closeBtn && isVisible(closeBtn)) {
    closeBtn.click();
    await sleep(300);
  } else {
    // fallback：ESC
    document.dispatchEvent(new KeyboardEvent('keydown', { key: 'Escape', bubbles: true }));
    document.dispatchEvent(new KeyboardEvent('keyup', { key: 'Escape', bubbles: true }));
    await sleep(300);
  }
}

function detectNeedLogin() {
  const url = location.href;
  if (textIncludesAny(url, ['login', 'sign-in', 'signin'])) return { hit: true, reason: 'url_login' };
  const bodyText = document.body ? document.body.innerText || '' : '';
  if (textIncludesAny(bodyText, ['log in', 'login', 'sign in', '请登录', '登录'])) return { hit: true, reason: 'text_login' };
  return { hit: false };
}

function detectRateLimited() {
  const url = location.href;
  const bodyText = document.body ? document.body.innerText || '' : '';
  const captchaEl = document.querySelector('iframe[src*="captcha"], iframe[src*="verify"], iframe[src*="challenge"], div[class*="captcha"], div[id*="captcha"], input[name*="captcha"]');
  if (captchaEl) return { hit: true, reason: 'captcha_dom', snippet: truncateText(captchaEl.innerText || captchaEl.getAttribute('src') || '', 200) };
  if (textIncludesAny(url, ['captcha', 'verify', 'challenge'])) return { hit: true, reason: 'url_challenge' };
  if (textIncludesAny(bodyText, ['too many requests', 'rate limit', '验证', '验证码', '操作过于频繁', '访问过于频繁'])) {
    return { hit: true, reason: 'text_rate_limit', snippet: truncateText(bodyText, 240) };
  }
  return { hit: false };
}

function detectNotFound(uniqueId) {
  const bodyText = document.body ? document.body.innerText || '' : '';
  if (textIncludesAny(bodyText, ['no results', 'not found', '0 results', '没有结果', '未找到', '无结果'])) {
    return { hit: true, reason: 'text_no_results', snippet: truncateText(bodyText, 240) };
  }
  // 有些页面会在结果区域显示空态
  const emptyEl = document.querySelector('[data-testid*="empty"], [class*="Empty"], [class*="empty"], div[role="status"]');
  if (emptyEl && textIncludesAny(emptyEl.innerText || '', ['no', 'empty', '未', '无', 'not'])) {
    return { hit: true, reason: 'dom_empty', snippet: truncateText(emptyEl.innerText, 240) };
  }
  // 如果页面里出现明确的“未找到 @xxx”
  if (uniqueId && textIncludesAny(bodyText, [uniqueId])) {
    const hint = document.querySelector('div,span,p');
    if (hint && textIncludesAny(hint.innerText || '', ['not found', '未找到', '无结果'])) {
      return { hit: true, reason: 'text_no_results_uid', snippet: truncateText(hint.innerText, 240) };
    }
  }
  return { hit: false };
}

function detectRestricted() {
  const bodyText = document.body ? document.body.innerText || '' : '';
  if (textIncludesAny(bodyText, ['restricted', 'suspended', 'banned', 'violat', '风险', '受限', '不可用', '违规', '冻结'])) {
    return { hit: true, reason: 'text_restricted', snippet: truncateText(bodyText, 240) };
  }
  return { hit: false };
}

function detectActive(uniqueId) {
  // 尽量不依赖 class：寻找结果卡片中的 @unique_id 或指向 /@xxx 的链接
  const uid = (uniqueId || '').replace(/^@/, '');
  if (!uid) return { hit: false };

  const links = Array.from(document.querySelectorAll('a[href]'));
  const hitLink = links.find(a => {
    const href = a.getAttribute('href') || '';
    const text = a.innerText || '';
    return href.includes('/@') && (href.toLowerCase().includes(`/@${uid.toLowerCase()}`) || text.toLowerCase().includes(uid.toLowerCase()));
  });
  if (hitLink) return { hit: true, reason: 'link_profile', snippet: truncateText(hitLink.innerText || hitLink.getAttribute('href') || '', 240) };

  const bodyText = document.body ? document.body.innerText || '' : '';
  if (bodyText && bodyText.toLowerCase().includes(uid.toLowerCase())) {
    // 避免误判：需要同时出现 @ 或“账号/用户”语义
    if (textIncludesAny(bodyText, [`@${uid}`.toLowerCase()]) || textIncludesAny(bodyText, ['user', 'creator', 'account', '主播', '用户', '账号'])) {
      return { hit: true, reason: 'text_contains_uid', snippet: truncateText(bodyText, 240) };
    }
  }

  return { hit: false };
}

function detectInviteFlowAvailable() {
  // 注意：该入口文案会随语言/版本变化；这里尽量覆盖更多可能。
  const stableOrCn = findInviteHostButton();
  if (stableOrCn) return true;
  const btn = findClickableByText(document, [
    '邀请主播', '邀请达人', '邀请创作者',
    '招募主播', '招募达人', '招募创作者',
    'Invite', 'Recruit'
  ]);
  return !!btn;
}

async function performSearch(uniqueId) {
  // 先做一次“就地查找”，找不到则尝试轻量导航到可用页面，再等元素出现
  let input = findSearchInputNow(document);
  if (!input) {
    await ensureSyncUiContext('search');
    input = await waitForElement(getSearchInputSelectors(), 30_000);
  }

  // 有些页面先渲染 combobox 容器，真正的输入框需要 click 后才出现
  if (input && !isEditableControl(input)) {
    try {
      input.click();
      await sleep(200);
    } catch {
      // ignore
    }
    const revived = findSearchInputNow(document);
    if (revived) input = revived;
  }

  input.focus();
  setNativeValue(input, String(uniqueId || '').replace(/^@/, ''));
  input.dispatchEvent(new KeyboardEvent('keydown', { key: 'Enter', bubbles: true }));
  input.dispatchEvent(new KeyboardEvent('keyup', { key: 'Enter', bubbles: true }));
}

async function processOne(item, settings, batchTraceId, batchIndex) {
  const traceId = (globalThis.crypto && crypto.randomUUID) ? crypto.randomUUID() : String(Date.now()) + '_' + Math.random();
  const uniqueId = item.unique_id || item.uniqueId || item.unique_id;

  const startedAt = new Date().toISOString();

  try {
    const login = detectNeedLogin();
    if (login.hit) {
      return {
        id: item.id,
        unique_id: uniqueId,
        trace_id: traceId,
        batch_index: batchIndex,
        tiktok_status_norm: 'need_login',
        error: null,
        tiktok_status_raw: {
          url: location.href,
          branch: 'need_login',
          reason: login.reason,
          ts: startedAt
        }
      };
    }

    await performSearch(uniqueId);

    const jitterMin = Number(settings?.jitterMinSec || 3);
    const jitterMax = Number(settings?.jitterMaxSec || 8);
    await sleep(randInt(jitterMin * 1000, jitterMax * 1000));

    const risk = detectRateLimited();
    if (risk.hit) {
      return {
        id: item.id,
        unique_id: uniqueId,
        trace_id: traceId,
        batch_index: batchIndex,
        tiktok_status_norm: 'rate_limited',
        error: null,
        tiktok_status_raw: {
          url: location.href,
          branch: 'rate_limited',
          reason: risk.reason,
          snippet: risk.snippet,
          ts: startedAt
        }
      };
    }

    const login2 = detectNeedLogin();
    if (login2.hit) {
      return {
        id: item.id,
        unique_id: uniqueId,
        trace_id: traceId,
        batch_index: batchIndex,
        tiktok_status_norm: 'need_login',
        error: null,
        tiktok_status_raw: {
          url: location.href,
          branch: 'need_login',
          reason: login2.reason,
          ts: startedAt
        }
      };
    }

    const restricted = detectRestricted();
    if (restricted.hit) {
      return {
        id: item.id,
        unique_id: uniqueId,
        trace_id: traceId,
        batch_index: batchIndex,
        tiktok_status_norm: 'restricted',
        error: null,
        tiktok_status_raw: {
          url: location.href,
          branch: 'restricted',
          reason: restricted.reason,
          snippet: restricted.snippet,
          ts: startedAt
        }
      };
    }

    const active = detectActive(uniqueId);
    if (active.hit) {
      return {
        id: item.id,
        unique_id: uniqueId,
        trace_id: traceId,
        batch_index: batchIndex,
        tiktok_status_norm: 'active',
        error: null,
        tiktok_status_raw: {
          url: location.href,
          branch: 'active',
          reason: active.reason,
          snippet: active.snippet,
          ts: startedAt
        }
      };
    }

    const nf = detectNotFound(uniqueId);
    if (nf.hit) {
      return {
        id: item.id,
        unique_id: uniqueId,
        trace_id: traceId,
        batch_index: batchIndex,
        tiktok_status_norm: 'not_found',
        error: null,
        tiktok_status_raw: {
          url: location.href,
          branch: 'not_found',
          reason: nf.reason,
          snippet: nf.snippet,
          ts: startedAt
        }
      };
    }

    // 如果无法判定，尽量区分 page_changed vs unknown_error
    // 只有当“邀请入口 + 搜索框”都不存在时，才认为是 page_changed（避免 placeholder/文案改动导致误熔断）
    const ui = detectSyncUiAvailable();
    if (!ui.search && !ui.invite) {
      return {
        id: item.id,
        unique_id: uniqueId,
        trace_id: traceId,
        batch_index: batchIndex,
        tiktok_status_norm: 'page_changed',
        error: null,
        tiktok_status_raw: {
          url: location.href,
          branch: 'page_changed',
          reason: 'no_invite_or_search_ui_after_query',
          ts: startedAt
        }
      };
    }

    return {
      id: item.id,
      unique_id: uniqueId,
      trace_id: traceId,
      batch_index: batchIndex,
      tiktok_status_norm: 'unknown_error',
      error: 'unable_to_classify',
      tiktok_status_raw: {
        url: location.href,
        branch: 'unknown_error',
        reason: 'unable_to_classify',
        snippet: truncateText(document.body?.innerText || '', 240),
        ts: startedAt
      }
    };
  } catch (e) {
    const msg = String(e && e.message ? e.message : e);
    const ui = detectSyncUiAvailable();
    const status = (ui.search || ui.invite) ? 'unknown_error' : 'page_changed';
    return {
      id: item.id,
      unique_id: uniqueId,
      trace_id: traceId,
      batch_index: batchIndex,
      tiktok_status_norm: status,
      error: msg,
      tiktok_status_raw: {
        url: location.href,
        branch: status,
        reason: 'exception',
        error: msg,
        ts: startedAt
      }
    };
  }
}

async function runBatch(items, settings, batchTraceId) {
  if (isRunning) {
    await log('warn', '已有批次在运行，忽略新批次', { batchTraceId });
    return;
  }

  isRunning = true;
  currentBatchTraceId = batchTraceId;
  currentIndex = 0;

  await log('info', '开始处理批次', { count: items.length, batchTraceId });

  try {
    // 先尽量把页面切到“可执行状态”（邀请入口 or 搜索框）
    const ctx = await ensureSyncUiContext();
    if (ctx.mode === 'none') {
      // 重要：这里不要把整批任务都回写为 page_changed（会导致大批量 FAILED）。
      // 更合理的策略：通知 service worker 进入暂停态并清空本地 batch，让服务端锁在 lock_minutes 后自然过期再重拉；
      // 同时上报足够多的“页面线索”用于更新选择器/路由。
      await log('error', '未检测到可用的搜索框或邀请入口（疑似页面结构/入口变更）', { url: location.href, reason: ctx.reason });

      const links = Array.from(document.querySelectorAll('a[href]'))
        .filter(isVisible)
        .slice(0, 40)
        .map(a => ({
          href: (a.getAttribute('href') || '').slice(0, 300),
          text: truncateText((a.innerText || a.textContent || '').replace(/\s+/g, ' ').trim(), 80)
        }))
        .filter(x => x.href || x.text);

      try {
        await chrome.runtime.sendMessage({
          type: 'TK_SYNC_BATCH_ABORTED',
          batchTraceId,
          reason: ctx.reason || 'no_invite_or_search_ui',
          url: location.href,
          title: document.title || '',
          // 降低敏感信息风险：只截取短文本摘要
          bodySnippet: truncateText(document.body?.innerText || '', 400),
          links
        });
      } catch (e) {
        await log('error', '发送 batch_aborted 到 service_worker 失败', { err: String(e) });
      }
      return;
    }

    // 优先走“邀请抽屉流程”，失败则自动降级到搜索框模式（避免误判 page_changed 直接熔断）
    if (ctx.mode === 'invite') {
      const chunkSize = 20;
      await log('info', '检测到邀请抽屉流程入口，启用批量查询模式', { chunkSize, ctxReason: ctx.reason });

      for (let i = 0; i < items.length; i += chunkSize) {
        if (!isRunning || currentBatchTraceId !== batchTraceId) break;
        currentIndex = i;

        const chunk = items.slice(i, i + chunkSize);
        const chunkIds = chunk.map(it => it.unique_id || it.uniqueId || it.unique_id).filter(Boolean);

        const startedAt = new Date().toISOString();
        try {
          const risk0 = detectRateLimited();
          if (risk0.hit) {
            // 直接把该 chunk 全部标记为 rate_limited（service worker 会触发 cooldown）
            for (let j = 0; j < chunk.length; j += 1) {
              const it = chunk[j];
              const uid = it.unique_id || it.uniqueId || it.unique_id;
              const res = {
                id: it.id,
                unique_id: uid,
                trace_id: (globalThis.crypto && crypto.randomUUID) ? crypto.randomUUID() : String(Date.now()) + '_' + Math.random(),
                batch_index: i + j,
                tiktok_status_norm: 'rate_limited',
                error: null,
                tiktok_status_raw: {
                  url: location.href,
                  branch: 'rate_limited_before_drawer',
                  reason: risk0.reason,
                  snippet: risk0.snippet,
                  ts: startedAt
                }
              };
              await chrome.runtime.sendMessage({ type: 'TK_SYNC_ITEM_RESULT', ...res });
            }
            continue;
          }

          await openInviteDrawer();
          const dlgAfterNext = await fillDrawerAndNext(chunkIds);

          const parsed = parseInviteResultsFromDrawer(dlgAfterNext, chunkIds);
          for (let j = 0; j < chunk.length; j += 1) {
            const it = chunk[j];
            const uid = it.unique_id || it.uniqueId || it.unique_id;
            const key = normalizeIdForInviteInput(uid);
            const hit = parsed.find(x => x.id === key);
            const norm = hit?.tiktok_status_norm || 'unknown_error';
            const snippet = hit?.snippet || null;

            const res = {
              id: it.id,
              unique_id: uid,
              trace_id: (globalThis.crypto && crypto.randomUUID) ? crypto.randomUUID() : String(Date.now()) + '_' + Math.random(),
              batch_index: i + j,
              tiktok_status_norm: norm,
              error: norm === 'unknown_error' ? 'unable_to_parse_invite_result' : null,
              tiktok_status_raw: {
                url: location.href,
                branch: 'invite_drawer_batch',
                input: key,
                snippet,
                ts: startedAt
              }
            };
            try {
              await chrome.runtime.sendMessage({ type: 'TK_SYNC_ITEM_RESULT', ...res });
            } catch (e) {
              await log('error', '发送结果到 service_worker 失败', { err: String(e), res });
            }
          }
        } catch (e) {
          // 抽屉流程失败：降级为“逐条搜索框模式”（不要直接 page_changed 熔断）
          const msg = String(e && e.message ? e.message : e);
          await log('warn', '邀请抽屉流程失败，降级为搜索框逐条模式', { err: msg, at: location.href });
          for (let j = 0; j < chunk.length; j += 1) {
            const it = chunk[j];
            try {
              const res = await processOne(it, settings, batchTraceId, i + j);
              await chrome.runtime.sendMessage({ type: 'TK_SYNC_ITEM_RESULT', ...res });
            } catch (err2) {
              await log('error', '降级搜索框模式：发送结果到 service_worker 失败', { err: String(err2) });
            }
          }
        } finally {
          await closeDrawerIfPossible();
        }

        // 批次节奏：每个 chunk 后做一点 jitter，降低风控
        const jitterMin = Number(settings?.jitterMinSec || 3);
        const jitterMax = Number(settings?.jitterMaxSec || 8);
        await sleep(randInt(jitterMin * 1000, jitterMax * 1000));
      }
    } else {
      await log('warn', '未检测到邀请入口，使用搜索框逐条模式', { url: location.href, ctxReason: ctx.reason });
      for (let i = 0; i < items.length; i += 1) {
        if (!isRunning || currentBatchTraceId !== batchTraceId) break;
        currentIndex = i;
        const item = items[i];
        const res = await processOne(item, settings, batchTraceId, i);
        try {
          await chrome.runtime.sendMessage({ type: 'TK_SYNC_ITEM_RESULT', ...res });
        } catch (e) {
          await log('error', '发送结果到 service_worker 失败', { err: String(e), res });
        }
      }
    }
  } finally {
    isRunning = false;
    currentBatchTraceId = null;
  }
}

chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  (async () => {
    if (!msg || typeof msg !== 'object') return;

    if (msg.type === 'TK_SYNC_START_BATCH') {
      const items = Array.isArray(msg.items) ? msg.items : [];
      const settings = msg.settings || {};
      const batchTraceId = msg.batchTraceId || String(Date.now());
      await runBatch(items, settings, batchTraceId);
      sendResponse({ ok: true });
      return;
    }

    if (msg.type === 'TK_SYNC_OPEN_INVITE_DRAWER') {
      const url = location.href;
      const onOverview = isOverviewPageUrl(url);
      await log('debug', '收到手动动作：打开“邀请主播”抽屉', { url, onOverview });

      if (!onOverview) {
        sendResponse({ ok: false, error: 'not_overview_page', url });
        return;
      }

      try {
        await openInviteDrawer();
        sendResponse({ ok: true });
      } catch (e) {
        const err = String(e && e.message ? e.message : e);
        await log('debug', '手动动作失败：无法打开邀请抽屉', { url, err });
        sendResponse({ ok: false, error: err });
      }
      return;
    }
  })();

  return true;
});
