async function send(msg) {
  return await chrome.runtime.sendMessage(msg);
}

function originPattern(url) {
  const u = new URL(url);
  return u.origin + '/*';
}

function fmtMs(ms) {
  if (!ms || ms <= 0) return '-';
  const sec = Math.ceil(ms / 1000);
  const m = Math.floor(sec / 60);
  const s = sec % 60;
  if (m <= 0) return `${s}s`;
  return `${m}m${s}s`;
}

function safeJson(x) {
  try {
    return JSON.stringify(x, null, 2);
  } catch {
    return String(x);
  }
}

function setTag(mode) {
  const el = document.getElementById('modeTag');
  el.textContent = mode || '-';
  el.className = 'tag ' + (mode || '').toLowerCase();
}

let lastSettings = null;

async function refresh() {
  const res = await send({ type: 'TK_SYNC_GET_STATE' });
  if (!res || !res.ok) return;

  const { state, settings } = res;
  lastSettings = settings || null;
  setTag(state.mode);

  document.getElementById('deviceId').textContent = state.deviceId || '-';

  const batchTotal = state.batch?.items?.length || 0;
  const batchIndex = state.batchIndex || 0;
  document.getElementById('batchProgress').textContent = batchTotal ? `${batchIndex}/${batchTotal}` : '-';

  const now = Date.now();
  const cdMs = (state.cooldownUntil || 0) - now;
  document.getElementById('cooldown').textContent = cdMs > 0 ? `${fmtMs(cdMs)}（${state.cooldownReason || '-'}）` : '-';

  const st = state.stats || {};
  document.getElementById('stats').textContent = `success=${st.success || 0}, restricted=${st.restricted || 0}, not_found=${st.not_found || 0}, retry=${st.retry || 0}, failed=${st.failed || 0}`;

  const logs = Array.isArray(state.logs) ? state.logs.slice(0, 20) : [];
  const logsEl = document.getElementById('logs');
  logsEl.innerHTML = '';
  for (const l of logs) {
    const div = document.createElement('div');
    div.className = 'log';
    div.textContent = `[${l.ts}] ${l.level}: ${l.message}` + (l.extra ? `\n${safeJson(l.extra)}` : '');
    logsEl.appendChild(div);
  }

  // copy
  document.getElementById('copyLogsBtn').onclick = async () => {
    const all = Array.isArray(state.logs) ? state.logs.slice(0, 200) : [];
    const text = all.map(l => `[${l.ts}] ${l.level}: ${l.message}` + (l.extra ? `\n${safeJson(l.extra)}` : '')).join('\n\n');
    await navigator.clipboard.writeText(text);
  };

  // 仅在缺少 SYSTEM_BASE host 权限时强调显示授权按钮
  const grantBtn = document.getElementById('grantBtn');
  if (state.mode === 'PAUSED' && state.cooldownReason === 'missing_host_permission') {
    grantBtn.style.display = 'inline-block';
    grantBtn.textContent = '授权 SYSTEM_BASE（必需）';
  } else {
    grantBtn.style.display = 'inline-block';
    grantBtn.textContent = '授权 SYSTEM_BASE';
  }
}

document.getElementById('startBtn').addEventListener('click', async () => {
  await send({ type: 'TK_SYNC_POPUP_START' });
  await refresh();
});

document.getElementById('stopBtn').addEventListener('click', async () => {
  await send({ type: 'TK_SYNC_POPUP_STOP', reason: 'manual' });
  await refresh();
});

document.getElementById('tickBtn').addEventListener('click', async () => {
  await send({ type: 'TK_SYNC_POPUP_TICK' });
  await refresh();
});

document.getElementById('grantBtn').addEventListener('click', () => {
  let base = (lastSettings && lastSettings.systemBase) ? String(lastSettings.systemBase).trim() : '';
  if (!base) {
    chrome.runtime.openOptionsPage();
    return;
  }
  let pattern;
  try {
    pattern = originPattern(base);
  } catch {
    chrome.runtime.openOptionsPage();
    return;
  }

  // 必须由“用户手势（点击）”触发：不要在这里 await 任何异步再去 request
  chrome.permissions.request({ origins: [pattern] }).then(async granted => {
    if (granted) {
      await send({ type: 'TK_SYNC_POPUP_TICK' });
      await refresh();
    } else {
      await refresh();
    }
  });
});

document.getElementById('openOptionsBtn').addEventListener('click', async () => {
  await chrome.runtime.openOptionsPage();
});

refresh();
setInterval(refresh, 1000);
