const STATE_KEY = 'tk_sync_state';
const SETTINGS_KEY = 'tk_sync_settings';

const DEFAULT_SETTINGS = {
  systemBase: 'https://tkradar.com/',
  token: '',
  region: '*',
  // 0 表示“不限”（仍会受服务端 cap 限制）
  pullLimit: 0,
  lockMinutes: 10,
  tickSeconds: 60,
  maxAttempts: 5,
  backoffMinutes: [2, 5, 15, 30, 120],
  jitterMinSec: 3,
  jitterMaxSec: 8,
  tiktokAdminUrl: 'https://live-backstage.tiktok.com/portal/overview',
  consecutiveRiskThreshold: 5,
  cooldownMinutes: 30
};

function nowMs() {
  return Date.now();
}

function isoNow() {
  return new Date().toISOString();
}

function clampInt(v, min, max, def) {
  const n = Number.parseInt(v, 10);
  if (Number.isNaN(n)) return def;
  return Math.max(min, Math.min(max, n));
}

function randomUUID() {
  if (globalThis.crypto && crypto.randomUUID) return crypto.randomUUID();
  // fallback (not cryptographically strong, but OK for device_id)
  return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, c => {
    const r = (Math.random() * 16) | 0;
    const v = c === 'x' ? r : (r & 0x3) | 0x8;
    return v.toString(16);
  });
}

async function getSettings() {
  const raw = (await chrome.storage.local.get(SETTINGS_KEY))[SETTINGS_KEY];
  return { ...DEFAULT_SETTINGS, ...(raw || {}) };
}

async function setSettings(patch) {
  const cur = await getSettings();
  const next = { ...cur, ...(patch || {}) };
  await chrome.storage.local.set({ [SETTINGS_KEY]: next });
  return next;
}

async function getState() {
  const raw = (await chrome.storage.local.get(STATE_KEY))[STATE_KEY];
  return (
    raw || {
      deviceId: `chrome_ext_${randomUUID()}`,
      mode: 'STOPPED',
      running: false,
      tabId: null,
      batch: null,
      batchIndex: 0,
      batchTraceId: null,
      cooldownUntil: 0,
      cooldownReason: null,
      consecutiveRisk: 0,
      consecutivePageChanged: 0,
      stats: {
        date: new Date().toISOString().slice(0, 10),
        success: 0,
        restricted: 0,
        not_found: 0,
        retry: 0,
        failed: 0
      },
      logs: [],
      lastTickAt: 0,
      lastActivityAt: 0
    }
  );
}

async function setState(patch) {
  const cur = await getState();
  const next = { ...cur, ...(patch || {}) };
  await chrome.storage.local.set({ [STATE_KEY]: next });
  return next;
}

async function addLog(level, message, extra) {
  const cur = await getState();
  const entry = {
    ts: isoNow(),
    level,
    message,
    extra: extra ? sanitizeExtra(extra) : undefined
  };
  const logs = Array.isArray(cur.logs) ? cur.logs.slice(0) : [];
  logs.unshift(entry);
  const capped = logs.slice(0, 200);
  await setState({ logs: capped, lastActivityAt: nowMs() });
}

function sanitizeExtra(extra) {
  try {
    const cloned = JSON.parse(JSON.stringify(extra));
    if (cloned && typeof cloned === 'object') {
      if (cloned.token) cloned.token = '***';
      if (cloned.Authorization) cloned.Authorization = '***';
      if (cloned.authorization) cloned.authorization = '***';
    }
    return cloned;
  } catch {
    return undefined;
  }
}

function computeNextRetryAt(attempts, backoffMinutes) {
  const seq = Array.isArray(backoffMinutes) && backoffMinutes.length ? backoffMinutes : [2, 5, 15, 30, 120];
  const idx = Math.max(0, Math.min(seq.length - 1, attempts - 1));
  return new Date(Date.now() + seq[idx] * 60_000).toISOString();
}

async function apiPost(path, body) {
  const settings = await getSettings();
  const base = (settings.systemBase || '').replace(/\/+$/, '');
  const url = `${base}${path}`;

  // 注意：chrome.permissions.request() 必须由“用户手势（点击）”触发。
  // service worker 无人值守循环里不能自动弹权限框，只能检测并提示用户去 Options 授权。
  try {
    await ensureHostPermissionOrThrow(url);
  } catch (e) {
    await pauseForMissingHostPermission(url, e);
    throw e;
  }

  const headers = { 'Content-Type': 'application/json' };
  if (settings.token && String(settings.token).trim()) {
    headers['Authorization'] = `Bearer ${String(settings.token).trim()}`;
  }

  const resp = await fetch(url, {
    method: 'POST',
    headers,
    body: JSON.stringify(body || {})
  });
  const text = await resp.text();
  let data;
  try {
    data = text ? JSON.parse(text) : null;
  } catch {
    data = { raw_text: text };
  }
  if (!resp.ok) {
    const err = new Error(`HTTP ${resp.status}: ${text?.slice?.(0, 500) || ''}`);
    err.status = resp.status;
    err.response = data;
    throw err;
  }
  return data;
}

function makeMissingHostPermissionError(originPattern) {
  const err = new Error(`missing host permission: ${originPattern}`);
  err.code = 'MISSING_HOST_PERMISSION';
  err.origin = originPattern;
  return err;
}

async function ensureHostPermissionOrThrow(url) {
  const origin = new URL(url).origin + '/*';
  const has = await chrome.permissions.contains({ origins: [origin] });
  if (has) return true;
  throw makeMissingHostPermissionError(origin);
}

async function pauseForMissingHostPermission(url, err) {
  const origin = (() => {
    try {
      return err?.origin || new URL(url).origin + '/*';
    } catch {
      return err?.origin || null;
    }
  })();

  const state = await getState();
  // 去重：避免每分钟重复打印同一条“请授权”的错误
  if (state && state.cooldownReason === 'missing_host_permission') return;

  await setState({
    mode: 'PAUSED',
    cooldownUntil: 0,
    cooldownReason: 'missing_host_permission',
    // 进入等待授权态时清空批次，避免一直卡在“处理中/回写失败”
    batch: null,
    batchIndex: 0,
    batchTraceId: null
  });

  await addLog('warn', 'SYSTEM_BASE 未授权 host_permissions（请在 Popup/Options 授权）', {
    url,
    origin,
    err: String(err)
  });
}

async function ensureTikTokTab() {
  const settings = await getSettings();
  const state = await getState();
  const targetUrl = settings.tiktokAdminUrl || 'https://live-backstage.tiktok.com/';

  if (state.tabId != null) {
    try {
      const tab = await chrome.tabs.get(state.tabId);
      if (tab && tab.id != null) {
        if (!tab.url || !tab.url.startsWith('http')) {
          await chrome.tabs.update(tab.id, { url: targetUrl, active: true });
        }
        return tab.id;
      }
    } catch {
      // ignore
    }
  }

  const tab = await chrome.tabs.create({ url: targetUrl, active: true });
  await setState({ tabId: tab.id });
  return tab.id;
}

function waitForTabComplete(tabId, timeoutMs) {
  return new Promise((resolve, reject) => {
    const t0 = Date.now();

    const timer = setInterval(async () => {
      try {
        const tab = await chrome.tabs.get(tabId);
        if (tab.status === 'complete') {
          cleanup();
          resolve(true);
          return;
        }
        if (Date.now() - t0 > timeoutMs) {
          cleanup();
          reject(new Error('tab load timeout'));
        }
      } catch (e) {
        cleanup();
        reject(e);
      }
    }, 300);

    function cleanup() {
      clearInterval(timer);
    }
  });
}

async function start() {
  const state = await getState();
  // Start 之前先做一次 host 权限检测：避免刚启动就 pull 报错/刷日志
  const origin = await getSystemBaseOriginPattern();
  if (!origin) {
    await setState({ running: false, mode: 'PAUSED', cooldownUntil: 0, cooldownReason: 'invalid_system_base', consecutiveRisk: 0, consecutivePageChanged: 0 });
    await addLog('error', 'System Base URL 未配置或不合法（请在 Options 填写）');
    return { ok: false, reason: 'invalid_system_base' };
  }

  const has = await chrome.permissions.contains({ origins: [origin] });
  if (!has) {
    await setState({
      running: true,
      mode: 'PAUSED',
      cooldownUntil: 0,
      cooldownReason: 'missing_host_permission',
      consecutiveRisk: 0,
      consecutivePageChanged: 0,
      batch: null,
      batchIndex: 0,
      batchTraceId: null
    });
    await addLog('warn', 'Start 已进入等待授权态：请先授权 SYSTEM_BASE host_permissions', { origin });
    return { ok: false, reason: 'missing_host_permission', origin };
  }

  await setState({ running: true, mode: 'RUNNING', cooldownUntil: 0, cooldownReason: null, consecutiveRisk: 0, consecutivePageChanged: 0 });
  await addLog('info', '已启动无人值守循环', { deviceId: state.deviceId });
  await tick('manual_start');
  return { ok: true };
}

async function stop(reason) {
  await setState({ running: false, mode: 'STOPPED', batch: null, batchIndex: 0, batchTraceId: null, cooldownUntil: 0, cooldownReason: reason || null, consecutiveRisk: 0, consecutivePageChanged: 0 });
  await addLog('info', '已停止', { reason: reason || null });
}

async function pauseCooldown(reason, minutes) {
  const until = Date.now() + minutes * 60_000;
  await setState({ mode: 'PAUSED', cooldownUntil: until, cooldownReason: reason || 'cooldown' });
  await addLog('warn', '进入全局暂停（cooldown）', { reason, until: new Date(until).toISOString() });
}

async function getSystemBaseOriginPattern() {
  const settings = await getSettings();
  const base = String(settings.systemBase || '').trim();
  if (!base) return null;
  try {
    return new URL(base).origin + '/*';
  } catch {
    return null;
  }
}

async function maybeResumeFromMissingHostPermission() {
  const state = await getState();
  if (!state?.running) return false;
  if (state.mode !== 'PAUSED' || state.cooldownReason !== 'missing_host_permission') return false;

  const origin = await getSystemBaseOriginPattern();
  if (!origin) return false;

  const has = await chrome.permissions.contains({ origins: [origin] });
  if (!has) return false;

  await setState({ mode: 'RUNNING', cooldownUntil: 0, cooldownReason: null });
  await addLog('info', '检测到 SYSTEM_BASE host_permissions 已授权，恢复运行', { origin });
  return true;
}

async function tick(source) {
  const settings = await getSettings();
  let state = await getState();

  if (!state.running) return;

  // 等待用户在 Options 授权 SYSTEM_BASE 后自动恢复
  if (state.mode === 'PAUSED' && state.cooldownReason === 'missing_host_permission') {
    const resumed = await maybeResumeFromMissingHostPermission();
    if (!resumed) return;
    state = await getState(); // 刷新状态，避免继续使用旧的 PAUSED state
  }

  const minIntervalMs = clampInt(settings.tickSeconds, 10, 3600, 60) * 1000;
  if (state.lastTickAt && Date.now() - state.lastTickAt < minIntervalMs && source !== 'manual_start') {
    return;
  }
  await setState({ lastTickAt: Date.now() });

  // 非 missing_host_permission 的暂停态：直接返回
  if (state.mode === 'PAUSED') {
    if (state.cooldownUntil && Date.now() < state.cooldownUntil) return;
    // cooldownUntil 过期则自动继续
    if (state.cooldownUntil && Date.now() >= state.cooldownUntil) {
      await setState({ mode: 'RUNNING', cooldownUntil: 0, cooldownReason: null });
      state = await getState();
    } else {
      return;
    }
  }

  // 若已有 batch 在跑，交给 content script 继续；这里只做“兜底超时检测”
  if (state.batch && Array.isArray(state.batch.items) && state.batchIndex < state.batch.items.length) {
    const idleMs = Date.now() - (state.lastActivityAt || 0);
    if (idleMs > 5 * 60_000) {
      await addLog('warn', '批次疑似卡住（超过 5 分钟无进展），将重置批次并重新拉取', { idleMs });
      await setState({ batch: null, batchIndex: 0, batchTraceId: null });
    }
    return;
  }

  // 拉取新一批
  let items = [];
  try {
    items = await apiPost('/api/ops/tk-sync/pull', {
      region: settings.region,
      // 0=不限（服务端仍有 cap，避免一次性返回过大）
      limit: clampInt(settings.pullLimit, 0, 5000, 0),
      device_id: state.deviceId,
      lock_minutes: clampInt(settings.lockMinutes, 1, 120, 10)
    });
  } catch (e) {
    // 缺失 host 权限时，apiPost 已经输出“请在 Options 授权”的明确日志，这里避免重复记录
    if (e && e.code === 'MISSING_HOST_PERMISSION') return;
    await addLog('error', 'pull 失败', { err: String(e) });
    return;
  }

  if (!Array.isArray(items) || items.length === 0) {
    await addLog('info', 'pull 返回空数组（暂无可处理数据）', {
      region: settings.region,
      limit: clampInt(settings.pullLimit, 0, 5000, 0),
      deviceId: state.deviceId
    });
    return;
  }

  const batchTraceId = randomUUID();
  await setState({
    batch: { items, startedAt: isoNow() },
    batchIndex: 0,
    batchTraceId
  });
  await addLog('info', '拉取到新批次', { count: items.length, batchTraceId });

  // 确保 TikTok tab
  let tabId;
  try {
    tabId = await ensureTikTokTab();
    await waitForTabComplete(tabId, 60_000);
  } catch (e) {
    await addLog('error', '无法打开/加载 TikTok 后台页面', { err: String(e) });
    await pauseCooldown('tiktok_tab_error', 5);
    return;
  }

  // 注入/确保 content script 已加载
  try {
    await chrome.scripting.executeScript({ target: { tabId }, files: ['content_script.js'] });
  } catch {
    // 如果已通过 content_scripts 注入，这里可能报错，忽略即可
  }

  // 发给 content script 执行
  try {
    const payload = {
      type: 'TK_SYNC_START_BATCH',
      batchTraceId,
      deviceId: state.deviceId,
      settings: {
        jitterMinSec: clampInt(settings.jitterMinSec, 1, 60, 3),
        jitterMaxSec: clampInt(settings.jitterMaxSec, 1, 120, 8)
      },
      items
    };
    await chrome.tabs.sendMessage(tabId, payload);
    await addLog('info', '已下发批次到 content_script', { tabId, count: items.length });
  } catch (e) {
    await addLog('error', '向 content_script 下发批次失败（URL pattern/权限/页面不匹配？）', { err: String(e), tabId });
    await pauseCooldown('content_script_missing', 10);
  }
}

async function handleItemResult(msg) {
  const state = await getState();
  const settings = await getSettings();

  if (state.mode === 'PAUSED' && state.cooldownReason === 'missing_host_permission') return;
  if (!state.batch || !Array.isArray(state.batch.items)) return;

  const { id, unique_id, tiktok_status_norm, tiktok_status_raw, error } = msg;
  const statusNorm = String(tiktok_status_norm || '').trim();

  let syncStatus;
  if (['active', 'restricted', 'not_found'].includes(statusNorm)) syncStatus = 'DONE';
  else if (statusNorm === 'page_changed') syncStatus = 'FAILED';
  else syncStatus = 'RETRY';

  const attemptsInc = syncStatus === 'RETRY' ? 1 : 0;
  const nextRetryAt = syncStatus === 'RETRY' ? computeNextRetryAt(1, settings.backoffMinutes) : null;

  try {
    await apiPost('/api/ops/tk-sync/report', {
      id,
      device_id: state.deviceId,
      trace_id: msg.trace_id || randomUUID(),
      tiktok_status_norm: statusNorm,
      tiktok_status_raw: tiktok_status_raw || null,
      sync_status: syncStatus,
      error: error || null,
      attempts_inc: attemptsInc,
      next_retry_at: nextRetryAt
    });
  } catch (e) {
    if (e && e.code === 'MISSING_HOST_PERMISSION') return;
    await addLog('error', 'report 失败（将等待下次批次/锁过期后重试）', { id, unique_id, err: String(e) });
  }

  // 更新统计
  const st = state.stats || {};
  const today = new Date().toISOString().slice(0, 10);
  const stats = st.date === today ? { ...st } : { date: today, success: 0, restricted: 0, not_found: 0, retry: 0, failed: 0 };
  if (statusNorm === 'active') stats.success += 1;
  else if (statusNorm === 'restricted') stats.restricted += 1;
  else if (statusNorm === 'not_found') stats.not_found += 1;
  else if (syncStatus === 'RETRY') stats.retry += 1;
  else stats.failed += 1;

  // 熔断：need_login/rate_limited 连续 N 条
  let consecutiveRisk = Number(state.consecutiveRisk || 0);
  if (statusNorm === 'need_login' || statusNorm === 'rate_limited') consecutiveRisk += 1;
  else consecutiveRisk = 0;

  // page_changed：需要多次确认再永久暂停（避免单次误判导致全局熔断）
  let consecutivePageChanged = Number(state.consecutivePageChanged || 0);
  if (statusNorm === 'page_changed') consecutivePageChanged += 1;
  else consecutivePageChanged = 0;

  const nextIndex = clampInt(msg.batch_index, 0, 10_000_000, state.batchIndex) + 1;

  await setState({
    batchIndex: nextIndex,
    stats,
    consecutiveRisk,
    consecutivePageChanged,
    lastActivityAt: nowMs()
  });

  await addLog('info', '处理完成并已回写', { id, unique_id, statusNorm, syncStatus, batchIndex: nextIndex });

  if (statusNorm === 'page_changed') {
    if (consecutivePageChanged >= 3) {
      await setState({ running: false, mode: 'PAUSED', cooldownUntil: Date.now() + 365 * 24 * 60 * 60_000, cooldownReason: 'page_changed' });
      await addLog('error', '检测到 page_changed：已触发全局熔断暂停（需要更新选择器/解析逻辑）', { consecutivePageChanged });
      return;
    }
    await addLog('warn', '检测到 page_changed（未达熔断阈值，将继续处理）', { consecutivePageChanged });
  }

  if (consecutiveRisk >= clampInt(settings.consecutiveRiskThreshold, 1, 50, 5)) {
    await pauseCooldown(statusNorm, clampInt(settings.cooldownMinutes, 1, 24 * 60, 30));
    await setState({ consecutiveRisk: 0 });
  }

  // 批次完成则清空，等待下次 tick 拉取下一批
  const batchTotal = state.batch.items.length;
  if (nextIndex >= batchTotal) {
    await addLog('info', '批次完成，等待下一轮 pull', { batchTotal });
    await setState({ batch: null, batchIndex: 0, batchTraceId: null });
  }
}

async function heartbeatTick() {
  const state = await getState();
  if (!state.running) return;
  if (state.mode === 'PAUSED' && state.cooldownReason === 'missing_host_permission') return;
  if (!state.batch || !Array.isArray(state.batch.items) || state.batch.items.length === 0) return;
  const ids = state.batch.items.map(it => it.id).filter(Boolean);
  if (!ids.length) return;
  try {
    await apiPost('/api/ops/tk-sync/heartbeat', {
      device_id: state.deviceId,
      ids,
      extend_minutes: 10
    });
  } catch (e) {
    if (e && e.code === 'MISSING_HOST_PERMISSION') return;
    await addLog('warn', 'heartbeat 失败（可忽略）', { err: String(e) });
  }
}

chrome.runtime.onInstalled.addListener(async () => {
  await getSettings();
  const state = await getState();
  await chrome.storage.local.set({ [STATE_KEY]: state });

  chrome.alarms.create('tk_sync_tick', { periodInMinutes: 1 });
  chrome.alarms.create('tk_sync_heartbeat', { periodInMinutes: 5 });

  await addLog('info', '扩展已安装/更新', { deviceId: state.deviceId });
});

chrome.alarms.onAlarm.addListener(async alarm => {
  if (alarm.name === 'tk_sync_tick') {
    await tick('alarm');
  }
  if (alarm.name === 'tk_sync_heartbeat') {
    await heartbeatTick();
  }
});

// 授权发生后尽快恢复（无需等到下一次 alarm）
chrome.permissions.onAdded?.addListener(() => {
  (async () => {
    const resumed = await maybeResumeFromMissingHostPermission();
    if (resumed) await tick('permission_added');
  })();
});

// 若用户撤销了 SYSTEM_BASE 权限：进入暂停态，避免继续 pull/report
chrome.permissions.onRemoved?.addListener(() => {
  (async () => {
    const state = await getState();
    if (!state?.running) return;
    const origin = await getSystemBaseOriginPattern();
    if (!origin) return;
    const has = await chrome.permissions.contains({ origins: [origin] });
    if (has) return;
    await setState({
      mode: 'PAUSED',
      cooldownUntil: 0,
      cooldownReason: 'missing_host_permission',
      batch: null,
      batchIndex: 0,
      batchTraceId: null
    });
    await addLog('warn', '检测到 SYSTEM_BASE host_permissions 被撤销，已暂停运行（请重新授权）', { origin });
  })();
});

chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  (async () => {
    if (!msg || typeof msg !== 'object') return;

    if (msg.type === 'TK_SYNC_POPUP_START') {
      const res = await start();
      sendResponse(res && typeof res === 'object' ? res : { ok: true });
      return;
    }

    if (msg.type === 'TK_SYNC_POPUP_STOP') {
      await stop(msg.reason || 'manual');
      sendResponse({ ok: true });
      return;
    }

    if (msg.type === 'TK_SYNC_POPUP_TICK') {
      await tick('popup');
      sendResponse({ ok: true });
      return;
    }

    if (msg.type === 'TK_SYNC_ITEM_RESULT') {
      await handleItemResult(msg);
      sendResponse({ ok: true });
      return;
    }

    if (msg.type === 'TK_SYNC_LOG') {
      await addLog(String(msg.level || 'info'), String(msg.message || ''), msg.extra);
      sendResponse({ ok: true });
      return;
    }

    if (msg.type === 'TK_SYNC_BATCH_ABORTED') {
      // content_script 报告：当前页面无法找到“搜索框/邀请入口”，无法继续执行。
      // 策略：暂停扩展并清空本地 batch（避免把整批任务回写为 FAILED/page_changed）。
      // 服务端锁将按 lock_minutes 自动过期，之后可重拉；用户修复 URL/登录态/页面后手动 Start 恢复。
      const until = Date.now() + 365 * 24 * 60 * 60_000; // 长暂停，避免无人值守时反复拉取/占锁
      await setState({
        running: false,
        mode: 'PAUSED',
        cooldownUntil: until,
        cooldownReason: 'page_ui_missing',
        batch: null,
        batchIndex: 0,
        batchTraceId: null,
        consecutiveRisk: 0,
        consecutivePageChanged: 0
      });
      await addLog('error', '页面缺少搜索框/邀请入口，已暂停（请更新 TikTok 后台 URL 或选择器/解析逻辑）', {
        batchTraceId: msg.batchTraceId,
        reason: msg.reason,
        url: msg.url,
        title: msg.title,
        bodySnippet: msg.bodySnippet,
        links: msg.links
      });
      sendResponse({ ok: true });
      return;
    }

    if (msg.type === 'TK_SYNC_SET_SETTINGS') {
      const next = await setSettings(msg.patch || {});
      sendResponse({ ok: true, settings: next });
      return;
    }

    if (msg.type === 'TK_SYNC_GET_STATE') {
      const st = await getState();
      const settings = await getSettings();
      sendResponse({ ok: true, state: st, settings });
      return;
    }
  })();

  return true; // keep message channel open for async
});
